#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "log.h"
#include "log_protocol.h"

#define _PATH "./pipe"
#define _SLEEP_TIME 20000

// Welcome MOTD
void motd(void) {
  printf(
      " ______ ______ ______ ______ ______ ______ ______ ______ \n"
      "|______|______|______|______|______|______|______|______|\n\n"
      "This is\n"
      " _     _____ _____ _____ _____ _____ _____\n"
      "| |   |  _  |  __ \\_   _|  ___/  ___|_   _|\n"
      "| |   | | | | |  \\/ | | | |__ \\ `--.  | |\n"
      "| |   | | | | | __  | | |  __| `--. \\ | |\n"
      "| |___\\ \\_/ / |_\\ \\ | | | |___/\\__/ / | |\n"
      "\\_____/\\___/ \\____/ \\_/ \\____/\\____/  \\_/\n\n"
      "Enjoy the ride !\n"
      " ______ ______ ______ ______ ______ ______ ______ ______ \n"
      "|______|______|______|______|______|______|______|______|\n\n"
      );
}

// Log pipe file
FILE *file_pipe;

void send_ctx(int data_length, int maximum_log_file_size, int reverse_order) {
  ssize_t written_elts;
  struct log_desc desc = {};

  desc.type = LOG_DESC_CONTEXT;
  desc.data_length = data_length;
  desc.maximum_log_file_size = maximum_log_file_size;
  desc.reverse_order = reverse_order;

  printf("> [out] context : sending %d bytes of data, maximum_log_file_size = "
      "%d, reverse_order = %d\n", data_length, maximum_log_file_size,
      reverse_order);

  // Write the context descriptor
  written_elts = fwrite((char *)&desc, sizeof(struct log_desc), 1, file_pipe);
  if (written_elts != 1) {
    ERROR("Failed to write the log rescriptor, wrote %d elements, "
        "instead of %d: %s\n", written_elts, 1,
        strerror(errno));
    exit(1);
  }
}

void send_data(int data_length, const char *buf, int end_of_file) {
  struct log_desc desc = {};
  ssize_t written_elts;

  desc.type = LOG_DESC_DATA;
  desc.data_length = data_length;
  desc.end_of_file = end_of_file;

  printf("> [out] data : sending %d bytes of data, end_of_file = %d\n",
      data_length, end_of_file);

  // Write the descriptor
  written_elts = fwrite((char *)&desc, sizeof(struct log_desc), 1, file_pipe);
  if (written_elts != 1) {
    ERROR("Failed to write log rescriptor, wrote %d elements, "
        "instead of %d: %s\n", written_elts, 1,
        strerror(errno));
    exit(1);
  }

  // Write the data
  if (data_length > 0) {
    written_elts = fwrite(&buf[0], 1, data_length, file_pipe);
    if (written_elts != data_length) {
      ERROR("Failed to write log rescriptor, wrote %d elements, "
          "instead of %d: %s\n", written_elts, data_length,
          strerror(errno));
      exit(1);
    }
  }

  // Flush it !
  fflush(file_pipe);
}

// void send_end(void) {
// ...
// }
//

int read_file(int major, char *buf, int size, int offset) {
  FILE *in;
  size_t read_elts;
  char filename[0x100];

  // Create filename and open the file
  sprintf(filename, "logs/%d.0.txt", major);
  in = fopen(&filename[0], "r");
  if (in == NULL) {
    ERROR("Failed to open file %s\n", &filename[0]);
    return 1;
  }

  // Seek in the file
  if (fseek(in, offset, SEEK_SET) != 0) {
    ERROR("Failed to seek file %s\n", &filename[0]);
    return 1;
  }

  // Read the file
  NOTE("Read data\n");
  read_elts = fread(&buf[0], 1, size, in);
  if (read_elts != size) {
    ERROR("Failed to read file %s\n", &filename[0]);
    return 1;
  }

  return 0;
}

const char log_dummy[] = "Dummy\n";

void pipe_open(void) {
  // Open pipe file
  NOTE("Attempting to open pipe file %s\n", _PATH);
  file_pipe = fopen(_PATH, "w");
  if (file_pipe == NULL) {
    ERROR("Failed to open pipe %s: %s\n", _PATH, strerror(errno));
    exit(1);
  }
}

void test1(void) {
  int i;
  NOTE("Test start\n");

  // Context 1
  send_ctx(0, 0x20, 0);

  for (i = 0; i < 8; i++) {
    send_data(strlen(log_dummy), &log_dummy[0], 0);
  }

  // Flush the data
  send_data(0, 0, 1);

  // Context 2
  send_ctx(0, 0x40, 0);

  // Data 1
  for (i = 0; i < 16; i++) {
    send_data(strlen(log_dummy), &log_dummy[0], 0);
  }

  // Flush the data
  send_data(0, 0, 1);

  NOTE("End of test\n");
}

// void test_end(void) {
//   // Close the server
//   NOTE("Close the server\n");
//   // end
//   send_end();
// }

void wait_user(void) {
  printf("Continue ?\n");
  getchar();
}

int main(int argc, char *argv[]) {
  // Init log
  log_init();
  log_level(LOG_LEVEL_DEBUG);
  // MOTD
  motd();
  // Open the server pipe
  pipe_open();
  // Run test 1
  NOTE("Test 1\n");
  test1();
  wait_user();
  // Close the test session by ending the server
  // no way if test5() which gets a shell
  // test_end();
  return 0;
}
