#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <stdint.h>
#include <setjmp.h>
#include <signal.h>

#include "log.h"
#include "log_protocol.h"

#define _PATH "./pipe"
#define _LOG_DIR "./logs"

// Global variables

// Log file buffer management
struct log_file {
  char buf[_LOG_MAX_FILE_SIZE];
  char dumb[0xb0]; // put here relevant values
  unsigned int minor;
  unsigned int major;
  unsigned int buf_size;
};
struct log_file log_file;

// Log file
FILE *file_out;

// Log file maximum size management
unsigned int maximum_log_file_size = _LOG_MAX_FILE_SIZE;
unsigned int reverse_order = 0;

// Fault tolerance
jmp_buf env = {{.__mask_was_saved = 1}};

// Welcome MOTD
void motd(void) {
  printf(
      " ______ ______ ______ ______ ______ ______ ______ ______ \n"
      "|______|______|______|______|______|______|______|______|\n\n"
      "This is\n"
      " _     _____ _____   _____ ___________ _   _ ___________\n"
      "| |   |  _  |  __ \\ /  ___|  ___| ___ \\ | | |  ___| ___ \\\n"
      "| |   | | | | |  \\/ \\ `--.| |__ | |_/ / | | | |__ | |_/ /\n"
      "| |   | | | | | __   `--. \\  __||    /| | | |  __||    /\n"
      "| |___\\ \\_/ / |_\\ \\ /\\__/ / |___| |\\ \\\\ \\_/ / |___| |\\ \\\n"
      "\\_____/\\___/ \\____/ \\____/\\____/\\_| \\_|\\___/\\____/\\_| \\_|\n\n"
      "Enjoy the ride !\n"
      " ______ ______ ______ ______ ______ ______ ______ ______ \n"
      "|______|______|______|______|______|______|______|______|\n\n"
      );
}

void file_out_inc() {
  char name[0x100];
  log_file.minor++;
  if (file_out != NULL) {
    fclose(file_out);
  }
  NOTE("Incrementing log_file.minor to %d\n", log_file.minor);
  snprintf(&name[0], 0x100, "%s/%d.%d.txt", _LOG_DIR, log_file.major,
      log_file.minor);
  NOTE("Attempting to open %s in write mode\n", &name[0]);
  file_out = fopen(&name[0], "w");
  if (file_out == NULL) {
    ERROR("Failed to open file %s: %s\n", &name[0], strerror(errno));
    exit(1);
  }
}

void file_out_close(void) {
  if (file_out != NULL) {
    NOTE("Closing already openned file_out\n");
    fclose(file_out);
  }
}

void init(void) {
  struct stat statbuf;
  // Stat named pipe file to know if it is already created
  if (stat(_PATH, &statbuf) == 0) {
    // File exists
    NOTE("Pipe file %s exists, remove it\n", _PATH);
    // The we remove it
    if (unlink(_PATH) != 0) {
      ERROR("Failed to remove %s file : %s\n", _PATH,
          strerror(errno));
      exit(1);
    }
  }
  // Create pipe file
  NOTE("Create pipe %s\n", _PATH);
  if (mknod(_PATH, S_IFIFO | 0600, 0) != 0) {
    ERROR("Failed to remove %s file : %s\n", _PATH,
        strerror(errno));
    exit(1);
  }
  // Create output folder if doesn't exists
  if (stat(_LOG_DIR, &statbuf) != 0) {
    NOTE("Output log dir %s does not exist, create it ...\n", _LOG_DIR);
    if (mkdir(_LOG_DIR, 0744) != 0) {
      ERROR("Failed to create %s log dir: %s\n", _LOG_DIR,
          strerror(errno));
      exit(1);
    }
  } else {
    NOTE("Output log dir %s is already present\n", _LOG_DIR);
  }
  // Initialize log file buffer size
  log_file.buf_size = 0;
  // Initialize maximim file size
  maximum_log_file_size = _LOG_MAX_FILE_SIZE;
  // File log counter
  log_file.major = 0;
  log_file.minor = -1;
  // Open file out
  file_out = NULL;
  file_out_inc();
}

void desc_print(struct log_desc *desc) {
  printf(">>> [in] ");
  switch (desc->type) {
    case LOG_DESC_CONTEXT:
      printf("ctx desc: ");
      break;
    case LOG_DESC_DATA:
      printf("data desc:");
      break;
    case LOG_DESC_END:
      printf("end desc: ");
      break;
    default:
      ERROR("Failed to parse the log descriptor, type %d unknown: ",
          desc->type);
      exit(1);
  }
  printf(" type(%d), data_length(%d), ", desc->type,
      desc->data_length);
  switch (desc->type) {
    case LOG_DESC_CONTEXT:
      printf("maximum_log_file_size(%d), ", desc->maximum_log_file_size);
      printf("reverse_order(%d)", desc->reverse_order);
      break;
    case LOG_DESC_DATA:
      printf("end_of_file(%d)", desc->end_of_file);
      break;
    case LOG_DESC_END:
      printf("<unused>");
      break;
    default:
      ERROR("Failed to parse the log descriptor, type %d unknown: ",
          desc->type);
      exit(1);
  }
  printf("\n");
}

void data_print(struct log_desc *desc, char *data) {
  printf(">>> data:\n");
  hexdump(((void *)data), desc->data_length);
}

void reverse(void) {
  char temp_buf[_LOG_MAX_FILE_SIZE] = {};
  memcpy(&temp_buf[0], log_file.buf, log_file.buf_size);
}

void run(void) {
  // Input file for client commands
  FILE *pipe;
  // Open pipe file
  NOTE("Attempting to open pipe file %s\n", _PATH);
  pipe = fopen(_PATH, "r");
  if (pipe == NULL) {
    ERROR("Failed to open pipe %s: %s\n", _PATH, strerror(errno));
    exit(1);
  }
  // Read loop !!!
  while (1) {
    char temp_buf[_LOG_MAX_FILE_SIZE];
    struct log_desc desc;
    ssize_t read_elts;
    // Save the sate in order to be able to recover after an eventual copy error
    if (setjmp(env)) {
      NOTE("Recovered from error !\n");
      NOTE("Current values of major %d, minor %d and buf_size %d\n",
          log_file.major, log_file.minor, log_file.buf_size);
    } else {
      NOTE("Set current state in order to recover from errors\n");
    }
    // STEP I : Read one chunk
    read_elts = fread(&desc, sizeof(struct log_desc), 1, pipe);
    if (read_elts != 1) {
      ERROR("Failed to receive a log rescriptor, received %d elements, "
          "instead of %d: %s\n", read_elts, 1, strerror(errno));
      exit(1);
    }
    // Parse the descriptor
    switch (desc.type) {
      case LOG_DESC_CONTEXT:
      case LOG_DESC_DATA:
      case LOG_DESC_END:
        break;
      default:
        ERROR("Failed to parse the log descriptor, type %d unknown: ",
            desc.type);
        exit(1);
    }
    desc_print(&desc);
    // STEP II : Read its data if any
    if (desc.data_length > 0) {
      if (desc.data_length > _LOG_MAX_FILE_SIZE) {
        ERROR("Single data length is bigger than _LOG_MAX_FILE_SIZE: %d\n",
            desc.data_length);
        exit(1);
      }
      NOTE("Current descriptor annonces %d bytes to read\n", desc.data_length);
      // Read chunck data
      read_elts = fread(&temp_buf[0], 1, desc.data_length, pipe);
      if (read_elts != desc.data_length) {
        ERROR("Failed to receive the first's data, receive %d chars, "
            "instead of %d: %s\n", read_elts, desc.data_length,
            strerror(errno));
        exit(1);
      }
      data_print(&desc, &temp_buf[0]);
    }
    // Exit if end descriptor
    if (desc.type == LOG_DESC_END) {
      NOTE("Leaving receive loop\n");
      break;
    }
    // STEP III : Execute descriptor
    switch (desc.type) {
      case LOG_DESC_CONTEXT:
        if (desc.maximum_log_file_size <= _LOG_MAX_FILE_SIZE) {
          NOTE("maximum_log_file_size = %d\n", desc.maximum_log_file_size);
          maximum_log_file_size = desc.maximum_log_file_size;
        } else {
          ERROR("Trying to update with a to big maximum_log_file_size : %d\n",
              desc.maximum_log_file_size);
          exit(1);
        }
        // Set reverse order
        // Change detection to prevent flood
        if (reverse_order != desc.reverse_order) {
          if (desc.reverse_order) {
            printf(">>> reverse order ACTIVATED\n");
          } else {
            printf(">>> reverse order DEACTIVATED\n");
          }
        }
        reverse_order = desc.reverse_order;
        break;
      case LOG_DESC_DATA:
        if (maximum_log_file_size < desc.data_length) {
          ERROR("%d byte log data can't fit in a any file of %d maximum size\n",
              desc.data_length, maximum_log_file_size);
          exit(1);
        }
        NOTE("State of buffer sizes before copy to log_file.buf:\n   "
            "maximum_log_file_size(%u), "
            "log_file.buf_size(%u)\n", maximum_log_file_size,
            log_file.buf_size);
        unsigned int remaining_room = maximum_log_file_size -
          log_file.buf_size;
        NOTE("Remaining room in log file buffer: %u (0x%08x)\n",
            remaining_room, remaining_room);
        // Flushing if necessary
        if (remaining_room < desc.data_length || desc.end_of_file == 1) {
          int written_elts;
          // Mangement of end of file
          if (desc.end_of_file == 1) {
            log_file.major++;
            log_file.minor=-1;
            NOTE("End of file is placed: increment log_file.major to %d\n",
                log_file.major);
          }
          // Apply reverse order
          if (reverse_order) {
            reverse();
            NOTE("Reversed !\n");
          }
          // Dump file buffer to the file system
          NOTE("Dumping %d bytes to log file\n", log_file.buf_size);
          written_elts = fwrite(log_file.buf, 1,
              log_file.buf_size, file_out);
          fflush(file_out);
          if (written_elts != log_file.buf_size) {
            ERROR("Failed to write log data, wrote %d elements, "
                "instead of %d: %s\n", written_elts, log_file.buf_size,
                strerror(errno));
            exit(1);
          }
          // Resetting log file buffer size
          log_file.buf_size = 0;
          // Openning next log file
          file_out_inc();
        }
        // Dumping to buffer
        memcpy(log_file.buf + log_file.buf_size, &temp_buf[0],
            desc.data_length);
        log_file.buf_size += desc.data_length;
        NOTE("New internal log file buffer size %d, %d room left\n",
            log_file.buf_size, maximum_log_file_size -
            log_file.buf_size);
        break;
      default:
        ERROR("Failed to parse the log descriptor, type %d unknown: ",
            desc.type);
        exit(1);
    }
    // ADD to file buffer or write to next file and reset buffer
    // Here is the integer overflow in the heap
    // and so on
  }
  // Eventually close the file
  NOTE("Eventually close pipe %s file\n", _PATH);
  fclose(pipe);
  file_out_close();
}

// Copy error handler
void error_handler(int sig) {
  signal(SIGABRT, error_handler);
  ERROR("Copy error : stack smashed.. go back to safe state: longjmp()\n");
  longjmp(env, -1);
}

// Init copy error handler
void init_error_recovery(void) {
  signal(SIGABRT, error_handler);
}

int main(int argc, char *argv[]) {
  // Init log
  log_init();
  log_level(LOG_LEVEL_DEBUG);
  // MOTD
  motd();
  // Remove if any and create pipe file
  init();
  // Error recovery init
  init_error_recovery();
  // Run the log server
  run();
  // Display motd function address
  NOTE("motd() address : %p\n", motd);
  return 0;
}
